<?php
/**
 * IZ_Integration_API
 *
 * @class           IZ_Integration_API
 * @since           1.0.0
 * @package         WC_iZettle_Integration
 * @category        Class
 * @author          bjorntech
 */

defined('ABSPATH') || exit;

if (!class_exists('IZ_Integration_API', false)) {

    class IZ_Integration_API
    {

        /**
         * Contains the API url
         * @access protected
         */
        protected $api_url = 'izettle.com';

        /**
         * get function.
         *
         * Performs an API GET request
         *
         * @access public
         * @param $path
         * @param $send_bearer
         * @param $url_prefix
         * @param $form
         * @return object
         */
        public function get($path, $send_bearer, $url_prefix = '', $form = array())
        {
            return $this->execute('GET', $path, $url_prefix, false, $form, $send_bearer);
        }

        /**
         * post function.
         *
         * Performs an API POST request
         *
         * @access public
         * @return object
         */
        public function post($path, $form = array(), $send_json = true, $send_bearer = true, $url_prefix = '')
        {
            return $this->execute('POST', $path, $url_prefix, $send_json, $form, $send_bearer);
        }

        /**
         * put function.
         *
         * Performs an API PUT request
         *
         * @access public
         * @return object
         */
        public function put($path, $form = array(), $send_json = true, $send_bearer = true, $url_prefix = '')
        {
            return $this->execute('PUT', $path, $url_prefix, $send_json, $form, $send_bearer);
        }

        /**
         * delete function.
         *
         * Performs an API DELETE request
         *
         * @access public
         * @return object
         */
        public function delete($path, $form = array(), $send_bearer = true, $url_prefix = '')
        {
            return $this->execute('DELETE', $path, $url_prefix, false, $form, $send_bearer);
        }

        /**
         * execute function.
         *
         * Executes the API request
         *
         * @access public
         * @param  string $request_type
         * @param  array  $form
         * @return object
         * @throws IZ_Integration_API_Exception
         */

        public function execute($request_type, $path, $url_prefix, $send_json, $form, $send_bearer)
        {

            if (($access_token = $this->get_access_token()) || ('token' == $path)) {

                $request_form_data = '';
                $params = '';

                if ('zettle' === $url_prefix) {
                    $api_url = $this->get_adm_url();
                } else {
                    $api_url = $this->api_url;
                }

                $url = trailingslashit('https://' . $url_prefix . '.' . $api_url . '/' . $path);

                $args = array(
                    'method' => $request_type,
                    'timeout' => 120,
                );

                if ($send_bearer === true) {
                    $args['headers'] = array(
                        'Content-Type' => 'application/json',
                        'Authorization' => 'Bearer ' . $access_token,
                    );
                } else {
                    $args['headers'] = $send_bearer;
                }

                if (is_array($form) && !empty($form)) {

                    if ('GET' == $request_type || 'DELETE' == $request_type) {
                        $url .= '?' . preg_replace('/%5B[0-9]+%5D/simU', '%5B%5D', http_build_query($form, '', '&'));
                    }

                    if (!$send_json) {
                        $args['body'] = $form;
                    } else {
                        $json_body = json_encode($form, JSON_INVALID_UTF8_IGNORE);
                        if (!$json_body) {
                            $json_error = json_last_error();
                            throw new IZ_Integration_API_Exception('JSON conversion failed when connecting to Zettle', $json_error, null, $url);
                        }
                        $args['body'] = $json_body;
                    }

                }

                if ('yes' == get_option('izettle_send_through_service') && 'zettle' !== $url_prefix) {

                    $args['headers'] = array(
                        'X-Url' => $url,
                        'X-Uuid' => $this->get_organization_uuid(),
                        'X-Website' => ($alternate_url = get_option('bjorntech_alternate_webhook_url')) ? $alternate_url : get_site_url(),
                    );

                    $response = wp_remote_request('https://zettle.' . $this->get_adm_url() . '/api', $args);

                } else {

                    $response = wp_remote_request($url, $args);

                }

                if (is_wp_error($response)) {
                    $code = $response->get_error_code();
                    $error = $response->get_error_message($code);
                    WC_IZ()->logger->add(sprintf('execute: %s - %s', $code, $error));
                    throw new IZ_Integration_API_Exception(sprintf('Got error %s when connecting to Zettle', $code), 0, null, $url);
                }

                $data = json_decode(wp_remote_retrieve_body($response));

                if (($http_code = wp_remote_retrieve_response_code($response)) > 299) {

                    if (isset($data->violations)) {
                        $violatons = reset($data->violations);
                        $message = $violatons->developerMessage;
                    } elseif (isset($data->developerMessage)) {
                        $message = $data->developerMessage;
                    } else {
                        $message = __('Unknown error when connecting to Zettle', 'woo-izettle-integration');
                    }

                    throw new IZ_Integration_API_Exception($message, $http_code);
                }

                return $data;

            } else {

                throw new IZ_Integration_API_Exception('Token error - Contact BjornTech support at hello@bjorntech.com to resolve');

            }

        }

    }
}
